<?php

namespace Tests\Feature;

use App\Enums\SituacaoEnum;
use App\Enums\StatusEnum;
use App\Enums\TipoPessoaTomador;
use App\Models\AtividadeEconomica;
use App\Models\DadosCadastrais;
use App\Models\Instituicao;
use App\Models\Lote;
use App\Models\NumeracaoNota;
use App\Models\Prestador;
use App\Models\PrestadorNotaFiscal;
use App\Models\Rps;
use App\Models\Tomador;
use App\Models\TomadorNotaFiscal;
use App\Services\AtividadeEconomica\CriaAtividadeEconomica;
use App\Services\DadosCadastrais\CriaDadosCadastrais;
use App\Services\Notas\BuscaNumeroDaNota;
use App\Services\Notas\CriadorNotaFiscalServico;
use App\Services\Notas\FormataCodigoDeVerificacaoNota;
use App\Services\Prestador\BuscaPrestador;
use App\Services\Prestador\CriaPrestador;
use App\Services\Prestador\CriaPrestadorNotaFiscal;
use App\Services\Tomador\CriaTomador;
use App\Services\Tomador\CriaTomadorNotaFiscal;
use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class CriacaoDeNotasTest extends TestCase
{
    use RefreshDatabase;

    private $instituicao;
    private Lote $lote;
    private array $rps = [];
    private Rps $rps1;

    private DadosCadastrais $dadosCadastraisSalvos;
    private Prestador $prestadorCriado;
    private AtividadeEconomica $atividadeEconomicaCriada;
    private Tomador $tomadorCriado;
    private TomadorNotaFiscal $tomadorNota;
    private PrestadorNotaFiscal $prestadorNotaCriado;

    public function test_criar_nota_fiscal_servico_com_tomador_ja_cadastrado()
    {
        $criadorDeNotas = new CriadorNotaFiscalServico($this->lote);
        $this->assertInstanceOf(CriadorNotaFiscalServico::class, $criadorDeNotas);

        $prestador_id = BuscaPrestador::buscarId($this->lote->cnpj);
    }

    public function test_busca_numeracao_pela_primeira_vez_para_nota_fiscal()
    {
        $prestador_id = BuscaPrestador::buscarId($this->lote->cnpj);
        $numero_nota = BuscaNumeroDaNota::buscarOuGerarSeNaoExistir($prestador_id);

        $this->assertEquals('0000001', $numero_nota);
    }

    public function test_busca_numeracao_ja_com_outras_notas_fiscais_cadastradas()
    {
        $prestador_id = BuscaPrestador::buscarId($this->lote->cnpj);

        $numeracaoNota2 = NumeracaoNota::create([
            'id_prestador' => $prestador_id,
            'numero' => 51
        ]);

        $numeracaoNota1 = NumeracaoNota::create([
            'id_prestador' => $prestador_id,
            'numero' => 53
        ]);

        $numero_nota = BuscaNumeroDaNota::buscarOuGerarSeNaoExistir($prestador_id);
        $numeracaoNotaQuandoExiste = BuscaNumeroDaNota::buscarApenasSeExistir($prestador_id);

        $this->assertEquals('0000054', $numero_nota);
        $this->assertEquals('0000053', $numeracaoNotaQuandoExiste);
    }

    /**
     * Retorna false se ainda não foi gerada nenhuma nota para o prestador.
     */
    public function test_buscar_numeracao_nota_quando_nao_existir_notas_cadasatradas_para_um_prestador()
    {
        $prestador_id = BuscaPrestador::buscarId('52001238000135');

        $numeroNotaFalseQuandoNaoExiste = BuscaNumeroDaNota::buscarApenasSeExistir($prestador_id);
        $numeroNotaGeradoSeNaoExistir = BuscaNumeroDaNota::buscarOuGerarSeNaoExistir($prestador_id);

        $this->assertEquals(false, $numeroNotaFalseQuandoNaoExiste);
        $this->assertEquals('0000001', $numeroNotaGeradoSeNaoExistir);
    }

    public function test_popula_instancias_de_objetos_para_nota_fiscal()
    {
        $this->assertInstanceOf(Instituicao::class, $this->instituicao);
        $this->assertInstanceOf(Lote::class, $this->lote);
        $this->assertArrayHasKey('id', ['id' => 10]);
        $this->assertInstanceOf(Rps::class, $this->rps1);

        $this->assertInstanceOf(DadosCadastrais::class, $this->dadosCadastraisSalvos);
        $this->assertDatabaseHas('dados_cadastrais', ['cnpj_cpf' => '52001238000135']);

        $this->assertInstanceOf(Prestador::class, $this->prestadorCriado);
        $this->assertDatabaseHas('prestador', ['cpf_representante' => '26105065035']);

        $this->assertInstanceOf(AtividadeEconomica::class, $this->atividadeEconomicaCriada);
        $this->assertDatabaseHas('atividade_economica', ['id_prestador' => $this->prestadorCriado->id]);
        $this->assertDatabaseHas('atividade_economica', ['id_atividade_economica_classe' => 348]);

        $this->assertInstanceOf(Tomador::class, $this->tomadorCriado);
        $this->assertDatabaseHas('tomador', ['id_dados_cadastrais' => $this->dadosCadastraisSalvos->id]);

        $this->assertInstanceOf(TomadorNotaFiscal::class, $this->tomadorNota);
        $this->assertDatabaseHas('tomador_nota_fiscal', ['email_tomador' => 'tomador@teste.com']);

        $this->assertInstanceOf(PrestadorNotaFiscal::class, $this->prestadorNotaCriado);
        $this->assertDatabaseHas('prestador_nota_fiscal', ['numero_prestador' => '3232113']);
        $this->assertDatabaseHas('prestador_nota_fiscal', ['email_prestador' => $this->dadosCadastraisSalvos->email]);
    }

    protected function setUp(): void
    {
        parent::setUp();

        //Insituição
        $this->instituicao = new Instituicao();
        $this->instituicao->nome = 'PREFEITURA MUNICIPAL DE CAJAZEIRAS';
        $this->instituicao->cnpj = '08923971000115';
        $this->instituicao->situacao = 1;
        $this->instituicao->datainclusao = Carbon::now();
        $this->instituicao->data_alteracao = Carbon::now();
        $this->instituicao->email = 'teste@teste';
        $this->instituicao->rua = 'werr';
        $this->instituicao->numero = 'fsf';
        $this->instituicao->bairro = 'sfsf';
        $this->instituicao->cep = 'fsf';
        $this->instituicao->complemento = 'fsff';
        $this->instituicao->telefone = 'fsf';
        $this->instituicao->logomarca = 'fsf';
        $this->instituicao->link_acesso_sistema = '';
        $this->instituicao->multa = 0;
        $this->instituicao->juros = 0;
        $this->instituicao->aliquota = 0;
        $this->instituicao->dia_vencimento = 10;
        $this->instituicao->tipo_boleto = 1;
        $this->instituicao->dia_limite_cancelamento = 11;
        $this->instituicao->id_cidade = 1290;
        $this->instituicao->save();

        //Lote
        $this->lote = new Lote();
        $this->lote->lote_rps_id = '123';
        $this->lote->numero = 111;
        $this->lote->cnpj = '52001238000135';
        $this->lote->inscricao_municipal = '321654987';
        $this->lote->quantidade_rps = 10;
        $this->lote->status = StatusEnum::NORMAL;
        $this->lote->instituicao_id = $this->instituicao->id;

        //Rps
        $this->rps['Id']                         = '10';
        $this->rps['NumeroRps']                  = '9999';
        $this->rps['Serie']                      = 'RPS';
        $this->rps['Tipo']                       = 1;
        $this->rps['DataEmissaoRps']             = Carbon::now();
        $this->rps['NaturezaOperacao']           = '0012487';
        $this->rps['RegimeEspecialTributacao']   = 3;
        $this->rps['OptanteSimplesNacional']     = 2;
        $this->rps['IncentivadorCultural']       = 2;
        $this->rps['Competencia']                = '202109';
        $this->rps['Status']                     = 1;

        //Serviços - Valores
        $this->rps['ValorServicos']              = 350.00;
        $this->rps['ValorDeducoes']              = 0.00;
        $this->rps['ValorPis']                   = 0.00;
        $this->rps['ValorCofins']                = 0.00;
        $this->rps['ValorInss']                  = 0.00;
        $this->rps['ValorIr']                    = 0.00;
        $this->rps['ValorCsll']                  = 0.00;
        $this->rps['ValorIssRetido']             = 0.00;
        $this->rps['ValorIss']                   = 17.50;
        $this->rps['OutrasRetencoes']            = 0.00;
        $this->rps['Aliquota']                   = 0.00;
        $this->rps['DescontoIncondicionado']     = 0.00;
        $this->rps['DescontoCondicionado']       = 0.00;

        //Serviços
        $discriminacao = "NS:12487; OS:50765; Placa:PDQ7484; KM:105159; Chassi:9BFZB55HXF8530201; SRIDS - FERRAMENTA DE DIAGNOSTICO IDS; ST - SUBST. TRIBUTARIA PECAS CONF. PROTOCOLO 61/12 E 62/12.Prop.: THIAGO SILVA BARROS";

        $this->rps['ItemListaServico']           = '1401';
        $this->rps['CodigoCnae']                 = '4520001';
        $this->rps['CodigoTributacaoMunicipio']  = '4520001';
        $this->rps['Discriminacao']              = $discriminacao;
        $this->rps['CodigoMunicipioServico']     = '2503704';

        //Tomador
        $this->rps['CpfCnpjTomador'] = '$cpfCnpj';
        $this->rps['RazaoSocialTomador']                = 'THIAGO SILVA BARROS (121322)';
        $this->rps['EnderecoTomador']                   = 'ANTONIO ALVES CONSERVA';
        $this->rps['NumeroEnderecoTomador']             = '1104';
        $this->rps['ComplementoTomador']                = 'CASA';
        $this->rps['BairroTomador']                     = 'NOSSA SRA DO PERPETUO SOCORRO';
        $this->rps['CodigoMunicipioTomador']            = '2612208';
        $this->rps['UfTomador']                         = 'PE';
        $this->rps['CepTomador']                        = '56000000';

        //Contato
        $this->rps['TelefoneTomador']                   = '(83) 98888-5555';
        $this->rps['EmailTomador']                      = 'tomador@teste.com';

        $this->rps1 = new Rps();
        $this->rps1->rps_id = $this->rps['Id'];
        $this->rps1->numero = $this->rps['NumeroRps'];
        $this->rps1->serie = $this->rps['Serie'];
        $this->rps1->tipo = $this->rps['Tipo'];
        $this->rps1->data_emissao = $this->rps['DataEmissaoRps'];
        $this->rps1->natureza_operacao = $this->rps['NaturezaOperacao'];
        $this->rps1->regime_especial_tributacao = $this->rps['RegimeEspecialTributacao'];
        $this->rps1->optante_simples_nacional = $this->rps['OptanteSimplesNacional'];
        $this->rps1->incentivador_cultural = $this->rps['IncentivadorCultural'];
        $this->rps1->competencia = $this->rps['Competencia'];
        $this->rps1->status = $this->rps['Status'];

        $this->rps1->valor_servicos = $this->rps['ValorServicos'];
        $this->rps1->valor_deducoes = $this->rps['ValorDeducoes'];
        $this->rps1->valor_pis = $this->rps['ValorPis'];
        $this->rps1->valor_cofins = $this->rps['ValorCofins'];
        $this->rps1->valor_inss = $this->rps['ValorInss'];
        $this->rps1->valor_ir = $this->rps['ValorIr'];
        $this->rps1->valor_csll = $this->rps['ValorCsll'];
        $this->rps1->valor_iss_retido = $this->rps['ValorIssRetido'];
        $this->rps1->valor_iss = $this->rps['ValorIss'];
        $this->rps1->outras_retencoes = $this->rps['OutrasRetencoes'];
        $this->rps1->aliquota = $this->rps['Aliquota'];
        $this->rps1->desconto_incondicionado = $this->rps['DescontoIncondicionado'];
        $this->rps1->desconto_condicionado = $this->rps['DescontoCondicionado'];

        $this->rps1->item_listaServico = $this->rps['ItemListaServico'];
        $this->rps1->codigo_cnae = $this->rps['CodigoCnae'];
        $this->rps1->codigo_tributacao_municipio = $this->rps['CodigoTributacaoMunicipio'];

        $this->rps1->discriminacao = $this->rps['Discriminacao'];
        $this->rps1->codigo_municipio = $this->rps['CodigoMunicipioServico'];

        $this->rps1->cpf_cnpj_tomador = $this->rps['CpfCnpjTomador'];
        $this->rps1->razao_social_tomador = $this->rps['RazaoSocialTomador'];
        $this->rps1->endereco_tomador = $this->rps['EnderecoTomador'];
        $this->rps1->numero_tomador = $this->rps['NumeroEnderecoTomador'];
        $this->rps1->complemento_tomador = $this->rps['ComplementoTomador'];
        $this->rps1->bairro_tomador = $this->rps['BairroTomador'];
        $this->rps1->codigo_municipal_tomador = $this->rps['CodigoMunicipioTomador'];
        $this->rps1->uF_tomador = $this->rps['UfTomador'];
        $this->rps1->cep_tomador = $this->rps['CepTomador'];
        $this->rps1->telefone_tomador = $this->rps['TelefoneTomador'];
        $this->rps1->email_tomador = $this->rps['EmailTomador'];
        $this->rps1->lote_id = $this->lote->id;
        $this->rps1->save();

        /**
         * Dados Cadastrais
         */
        $criadorDeDadosCadastrais = new CriaDadosCadastrais(
            '52001238000135',
            'PRESTADOR TESTE WEBSERVICE',
            '16.139.962-3',
            '0020222',
            1058,
            '58900971',
            '',
            'Centro',
            'Rua do Comércio',
            's/n',
            '',
            'testewebservice@test.com',
            '(83) 98877-6655',
            '',
            TipoPessoaTomador::PESSOA_JURIDICA,
            'Manoel Proprietário',
            1290,
        );

        $this->dadosCadastraisSalvos = $criadorDeDadosCadastrais->criar();

        /**
         * Prestador
         */
        $criadorDePrestador = new CriaPrestador(
            '3232126546466',
            'REPRESENTANTE TESTE',
            '26105065035',
            $this->dadosCadastraisSalvos->id
        );

        $this->prestadorCriado = $criadorDePrestador->criar();

        /**
         * Prestador Nota Fiscal
         */
        $criadorDePrestadoresNota = new CriaPrestadorNotaFiscal(
            $this->dadosCadastraisSalvos->razao_nome,
            $this->dadosCadastraisSalvos->cep,
            'Cajazeiras/PARAÍBA', /** Buscar a cidade dinâmicamente quando em prod. */
            $this->dadosCadastraisSalvos->bairro,
            $this->dadosCadastraisSalvos->rua,
            '3232113',
            '',
            $this->dadosCadastraisSalvos->email,
            $this->dadosCadastraisSalvos->telefone,
            '25',
            TipoPessoaTomador::PESSOA_JURIDICA
        );

        $this->prestadorNotaCriado = $criadorDePrestadoresNota->criar();

        /**
         * Atividade Econômica
         */
        $criadorDeAtividadeEconomica = new CriaAtividadeEconomica(
            1,
            $this->prestadorCriado->id,
            348,
            664,
            1
        );

        $this->atividadeEconomicaCriada = $criadorDeAtividadeEconomica->criar();

        /**
         * Tomador
         */
        $criadorDeTomadores = new CriaTomador(
            $this->dadosCadastraisSalvos->id,
            $this->prestadorCriado->id,
            40 /** cajazeiras */
        );

        $this->tomadorCriado = $criadorDeTomadores->criar();

        /**
         * Tomador Nota Fiscal
         */
        $criadorDeTomadorNota = new CriaTomadorNotaFiscal(
            $this->dadosCadastraisSalvos->razao_nome,
            '82779205040',
            '58900970',
            'Centro',
            'Rua Coronel Juvêncio Carneiro',
            '489',
            '',
            'tomador@teste.com',
            '(83) 3333-4444',
            '',
            $this->dadosCadastraisSalvos->inscricao_estadual,
            $this->dadosCadastraisSalvos->inscricao_municipal,
            '',
            TipoPessoaTomador::PESSOA_JURIDICA,
            1058,
            1290
        );

        $this->tomadorNota = $criadorDeTomadorNota->criar();

    }
}
