<?php

namespace App\Services\Tomador;

use App\Models\DadosCadastrais;
use App\Models\Rps;
use App\Models\Tomador;
use App\Models\TomadorNotaFiscal;
use App\Services\DadosCadastrais\BuscaDadosCadastrais;
use App\Services\DadosCadastrais\CriaDadosCadastrais;
use Exception;

class BuscaCriaTomador
{
    /**
     * Busca o Tomador se já estiver cadastrado.
     * Se existir um Tomador, retorna um array contendo um objeto Tomador.
     * Se não existir, adiciona os dados cadastrais do tomador, adiciona o tomador nota fiscal e,
     * por fim, cadastra o novo tomador. Retorna um array com o Tomador e o TomadorNotaServico.
     *
     * @param Rps $rps
     * @param int $id_prestador
     * @param int $id_instituicao
     * @return array
     * @throws Exception
     */
    public function buscar(Rps $rps, int $id_prestador, int $id_instituicao) : array
    {
        try {

            $dadosCadastrais = BuscaDadosCadastrais::porCnpjOuCpf($rps->cpf_cnpj_tomador);

            if (!$dadosCadastrais) {

                $dadosCadastraisCriados = $this->criarDadosCadastrais($rps);

                $tomadorNotaFiscal = $this->criarTomadorNotaFiscal($rps);

                $tomador = $this->criarTomador($dadosCadastraisCriados->id, $id_prestador, $id_instituicao);

                return [ 'tomador' => $tomador, 'tomadorNotaServico' => $tomadorNotaFiscal ];
            }

            $tomador = Tomador::where('id_dados_cadastrais', $dadosCadastrais->id)
                ->where('id_prestador', $id_prestador)
                ->where('id_instituicao', $id_instituicao)
                ->first();

            if (!$tomador)
                $tomador = $this->criarTomador($dadosCadastrais->id, $id_prestador, $id_instituicao);

            $tomadorNotaFiscal = $this->criarTomadorNotaFiscal($rps);

            return ['tomador' => $tomador, 'tomadorNotaServico' => $tomadorNotaFiscal ];

        } catch (Exception $exception) {
            throw new Exception("Não foi possível encontrar ou cadastrar o Tomador: " . $exception->getMessage() );
        }
    }

    /**
     * @param Rps $rps
     * @return CriaDadosCadastrais
     * @throws Exception
     */
    private function criarDadosCadastrais(Rps $rps): DadosCadastrais
    {
        try {

            $cidade = BuscaCidade::peloCodigo($rps->codigo_municipal_tomador);

            $criadorDados = new CriaDadosCadastrais(
                $rps->cpf_cnpj_tomador,
                $rps->razao_social_tomador,
                "", // ?? // tem que alterar o xml para conter essa inforamção
                "", // ?? // tem que alterar o xml para conter essa inforamção
                1058,
                $rps->cep_tomador,
                "",
                $rps->bairro_tomador,
                $rps->endereco_tomador,
                $rps->numero_tomador,
                $rps->complemento_tomador,
                $rps->email_tomador,
                $rps->telefone_tomador,
                "",
                1, // tem que alterar o xml para conter essa inforamção
                $rps->telefone_tomador,
                $cidade->id
            );

            return $criadorDados->criar();

        } catch (Exception $exception) {
            throw new Exception($exception);
        }
    }

    /**
     * @param Rps $rps
     * @return TomadorNotaFiscal
     * @throws Exception
     */
    private function criarTomadorNotaFiscal(Rps $rps) : TomadorNotaFiscal
    {
        try {

            $cidade = BuscaCidade::peloCodigo($rps->codigo_municipal_tomador);

            $tipo_pessoa = strlen($rps->cpf_cnpj_tomador) === 14 ? 2 : 1;

            $tomadorNF = new CriaTomadorNotaFiscal(
                $rps->razao_social_tomador,
                $rps->cpf_cnpj_tomador,
                $rps->cep_tomador,
                $rps->bairro_tomador,
                $rps->endereco_tomador,
                $rps->numero_tomador,
                $rps->complemento_tomador,
                $rps->email_tomador,
                $rps->telefone_tomador,
                "",
                "", // tem que alterar o xml para conter essa inforamção
                "", // tem que alterar o xml para conter essa inforamção
                "",
                $tipo_pessoa, // tem que alterar o xml para conter essa inforamção
                1058, // Brasil
                $cidade->id
            );

            return $tomadorNF->criar();

        } catch (Exception $exception) {
            throw new Exception($exception);
        }
    }

    /**
     * @throws Exception
     */
    private function criarTomador($dadosCadastraisId, $prestadorId, $instituicaoId)
    {
        try {

            $criadorTomador = new CriaTomador(
                $dadosCadastraisId,
                $prestadorId,
                $instituicaoId
            );

            return $criadorTomador->criar();

        } catch (Exception $exception) {
            throw new Exception($exception);
        }
    }
}
