<?php

namespace App\Imports\CSV;

use App\Models\InstituicaoFinanceira\LancamentosFinanceiro;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ImportCSV
{
    private array $planilha;
    private string $csv_path_stored;
    private Model $balancete;

    public function __construct(string $csv_path_stored, Model $balanceteFinanceiro)
    {
        $this->csv_path_stored = $csv_path_stored;
        $this->balancete = $balanceteFinanceiro;
    }

    /**
     * @return int
     * @throws Exception
     */
    public function loadCsvPath() : int
    {
        try {

            $csv_file = Storage::path($this->csv_path_stored);
            $read = IOFactory::load($csv_file);

            $memoriaDisponivel = ini_get('memory_limit');
            Log::info("Memória disponível no sistema: " . $memoriaDisponivel);
//            dd($memoriaDisponivel);
            ini_set( 'memory_limit', '1024M' );
            $this->planilha = $read->getActiveSheet()->toArray();
            ini_set( 'memory_limit', $memoriaDisponivel );
            Log::info("Memória restaurada para a quantidade disponível no sistema");

            if (is_null($this->planilhaTemFormatoCorreto())) {
                throw new Exception("ImportCSV: O arquivo enviado não possui os dados corretos para um arquivo válido.");
            }

            $this->excluirLancamentosAnterioresDoBalanceteCasoExistam();

            return $this->storeLancamentos();

        } catch (Exception $exception) {
            throw new Exception("ImportCSV::class - loadCsvPath :" . $exception);
        }
    }

    /**
     * @return mixed
     * @throws Exception
     */
    private function planilhaTemFormatoCorreto(): mixed
    {
        try {
            foreach ($this->planilha as $linha => $colunas) {

                if (!empty($colunas[0])
                    && $colunas[0] === "Empresa"
                    && $colunas[1] === "Agência"
                    && $colunas[2] === "Nome da Agência"
                    && $colunas[3] === "CNPJ"
                    && $colunas[4] === "COSIF"
                    && $colunas[5] === "Conta"
                    && $colunas[6] === "Descrição da Conta"
                    && $colunas[7] === "Sal. Anterior"
                    && $colunas[8] === "Mov. Mês"
                    && $colunas[9] === "Sal. Semestre"
                    && $colunas[10] === "Sal. Exercício"
                    && $colunas[11] === "Código Serviço"
                    && $colunas[12] === "Alíquota"
                    && $colunas[13] === "Cálculo ISS - Mês") {

                    return $linha;
                }
            }

            return null;

        } catch (Exception $exception) {
            throw new Exception("ImportCSV::class - planilhaTemFormatoCorreto :" . $exception);
        }
    }

    /**
     * @throws Exception
     * @return int $qtdeLancamentosSalvos
     */
    private function storeLancamentos(): int
    {
        try {

            $qtdeLancamentosSalvos = 0;

            $linhaDoCabecalho = $this->planilhaTemFormatoCorreto();

            foreach ($this->planilha as $linha => $celulas) {

                if ($linha > $linhaDoCabecalho && !empty($celulas[0])) {

                    DB::beginTransaction();

                    LancamentosFinanceiro::create([
                        'cosif' => $celulas[4],
                        'conta' => $celulas[5],
                        'descricao_da_conta' => $celulas[6],
                        'saldo_anterio' => realToDatabase($celulas[7]),
                        'movimentacao_mes' => realToDatabase($celulas[8]),
                        'saldo_semestre' => realToDatabase($celulas[9]),
                        'saldo_exercicio' => realToDatabase($celulas[10]),
                        'codigo_serviço' => $celulas[11],
                        'aliquota' => realToDatabase($celulas[12]),
                        'calculo_iss_mes' => realToDatabase($celulas[13]),


//                        'saldo_anterio' => ajustaValorParaDBImportExcel($celulas[7]),
//                        'movimentacao_mes' => ajustaValorParaDBImportExcel($celulas[8]),
//                        'saldo_semestre' => ajustaValorParaDBImportExcel($celulas[9]),
//                        'saldo_exercicio' => ajustaValorParaDBImportExcel($celulas[10]),
//                        'codigo_serviço' => $celulas[11],
//                        'aliquota' => ajustaValorParaDBImportExcel($celulas[12]),
//                        'calculo_iss_mes' => ajustaValorParaDBImportExcel($celulas[13]),

                        'balancete_id' => $this->balancete->id,
                        'prestador_id' => $this->balancete->prestador->id,
                    ]);

                    DB::commit();

                    $qtdeLancamentosSalvos++;
                }
            }

            return $qtdeLancamentosSalvos;

        } catch (Exception $exception) {
            DB::rollBack();
            throw new Exception("ImportCSV::class - storeLancamentos :" . $exception);
        }
    }

    /**
     * @return void
     */
    private function excluirLancamentosAnterioresDoBalanceteCasoExistam() : void
    {
        if (count($this->balancete->lancamentos) > 0) {

            foreach ($this->balancete->lancamentos as $lancamento) {
                $lancamento->delete();
            }

        }
    }
}
