<?php

namespace App\Http\Livewire\BalanceteFinanceiro;

use App\Enums\MesEnum;
use App\Models\InstituicaoFinanceira\BalancetePrestador;
use App\Services\GeradorDeAnoMesDiasHorasEMinutos;
use App\Validadores\Balancetes\AplicaRegrasParaAtualizarBalancete;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Livewire\Component;
use Exception;
use RuntimeException;

class UpdateBalanceteModal extends Component
{
    public $showForm;
    public $alert;

    public $ano_competencia;
    public $mes_competencia;
    public $historico;

    public $uuid;
    public $prestador_ultima_atualizacao_id;

    protected $listeners = ['abrirEditBalanceteModal'];
    protected $rules = [
        'mes_competencia' => 'required|string',
        'ano_competencia' => 'required',
    ];

    public function render()
    {
        $anos = GeradorDeAnoMesDiasHorasEMinutos::gerarAnos();
        $meses = MesEnum::cases();

        return view('livewire.balancete-financeiro.update-balancete-modal', compact('anos', 'meses'));
    }

    public function abrirEditBalanceteModal(string $balancete_uuid) : void
    {
        $this->alert = [];
        $this->showForm = true;

        $this->uuid = $balancete_uuid;

        $balancete = BalancetePrestador::where('uuid', $this->uuid)->first();

        $this->mes_competencia = $balancete->mes_competencia;
        $this->ano_competencia = $balancete->ano_competencia;
        $this->historico = $balancete->historico;
    }

    public function submit(): void
    {
        $this->validate();
        sleep(1);
        $this->updateBalancete();
    }

    private function updateBalancete(): void
    {
        try {

            $prestador = Session::get('prestador');
            $prestador['instituicao'] = Session::get('instituicao');

            $balancete_log_antes = BalancetePrestador::where('uuid', $this->uuid)->first();

            $balancete = [
                'id'                                => $balancete_log_antes->id,
                'mes_competencia'                   => $this->mes_competencia,
                'ano_competencia'                   => $this->ano_competencia,
                'historico'                         => $this->historico,
                'prestador_ultima_atualizacao_id'   => $prestador['id'],
            ];

            $balancete_para_validar = $balancete;
            $balancete_para_validar['prestador'] = $prestador;

            (new AplicaRegrasParaAtualizarBalancete())->aplicar($balancete_para_validar);

            DB::beginTransaction();

            BalancetePrestador::where('uuid', $this->uuid)
                ->update($balancete);

            $balancete_log_depois = BalancetePrestador::where('uuid', $this->uuid)->first();

            DB::commit();

            Log::info("Balancete Financeiro de ID {$balancete_log_depois->id} atualizado pelo Prestador de id {$prestador['id']}", [
                'antes' => $balancete_log_antes,
                'depois' => $balancete_log_depois,
            ]);

            $this->showForm = false;

            $this->alert = [
                'show' => true,
                'type' => 'success',
                'title' => 'Sucesso!',
                'description' => "Balancete " . MesEnum::from($balancete_log_depois->mes_competencia)->name ."/{$balancete_log_depois->ano_competencia} Atualizado com sucesso."
            ];

            $this->emit('refreshDatatable');

        } catch (Exception|RuntimeException $exception) {

            DB::rollBack();
            Log::error("Atualizando Balancete Financeiro: " . $exception->getMessage());

            $exception_error = ($exception instanceof \RuntimeException)
                ? $exception->getMessage()
                : "Houve um problema durante o procedimento e não foi possível atualizar o balancete ". MesEnum::from($this->mes_competencia)->name ."/".$this->ano_competencia;

            $this->alert = [
                'show' => true,
                'type' => 'danger',
                'title' => 'Erro!',
                'description' => $exception_error
            ];
        }
    }
}
