<?php

namespace App\Http\Livewire\BalanceteFinanceiro;

use App\Enums\MesEnum;
use App\Models\InstituicaoFinanceira\BalancetePrestador;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Livewire\Component;
use PHPUnit\Exception;

class DeleteBalanceteModal extends Component
{
    public $alert;
    public $showFormButtons;
    public $balancete;
    public $motivo_exclusao;

    protected $listeners = ['abrirDeleteBalanceteModal'];

    protected $rules = [
        'motivo_exclusao' => 'required',
    ];
    public function render()
    {
        return view('livewire.balancete-financeiro.delete-balancete-modal');
    }

    public function abrirDeleteBalanceteModal(string $uuid): void
    {
        $this->balancete = BalancetePrestador::with('lancamentos')
            ->where('uuid', $uuid)
            ->first();

        $this->showFormButtons = true;
        $this->motivo_exclusao = "";
    }

    public function excluirBalancete() : void
    {
        $balancete_log_a_excluir = $this->balancete;

        $this->validate();

        try {

            DB::beginTransaction();

            $this->excluirLancamentosDoBalancete();

            $prestador_id = Session::get('prestador')->id;

            $this->balancete->motivo_exclusao = $this->motivo_exclusao;
            $this->balancete->prestador_exclusao_id = $prestador_id;
            $this->balancete->save();

            $this->balancete->delete();

            $prestador_id = Session::get('prestador')->id;

            Log::info("Balancete Financeiro Excluído pelo Prestador de id {$prestador_id}", $balancete_log_a_excluir->toArray());

            DB::commit();

            $this->showFormButtons = false;

            $this->alert = [
                'show' => true,
                'type' => 'success',
                'title' => 'Processo de exclusão concluído!',
                'description' => "Balancete " . MesEnum::from($balancete_log_a_excluir->mes_competencia)->name ."/{$balancete_log_a_excluir->ano_competencia} excluído com sucesso."
            ];

            $this->emit('refreshDatatable');

        } catch (Exception $exception) {
            DB::rollBack();
            Log::error("Exclusão Balancete Financeiro: " . $exception->getMessage());

            $this->alert = [
                'show' => true,
                'type' => 'danger',
                'title' => 'Erro!',
                'description' => "Houve um problema durante o procedimento e não foi possível excluir o balancete "
                    . MesEnum::from($balancete_log_a_excluir->mes_competencia)->name ."/". $balancete_log_a_excluir->ano_competencia
            ];
        }
    }

    /**
     * Se o balancete possuir lançamentos, estes lançamentos deverão ser excluídos antes do balancete.
     * @return void
     */
    private function excluirLancamentosDoBalancete() : void
    {
        if ($this->balancete->lancamentos->count() > 0) {
            foreach ($this->balancete->lancamentos as $lancamento) {
                $lancamento->delete();
            }
        }
    }
}
