<?php

namespace App\Http\Livewire\BalanceteFinanceiro;

use App\Enums\AplicaMultaBalanceteEnum;
use App\Enums\MesEnum;
use App\Models\InstituicaoFinanceira\BalancetePrestador;
use App\Services\GeradorDeAnoMesDiasHorasEMinutos;
use App\Validadores\Balancetes\AplicaRegrasParaCadastroBalancete;
use BaconQrCode\Exception\RuntimeException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Livewire\Component;
use Exception;

class CreateBalanceteModal extends Component
{
    public $showForm;
    public $alert;

    public $ano_competencia;
    public $mes_competencia;
    public $historico;
    public $prestador;

    protected $listeners = ['abrirCreateBalanceteModal'];

    protected $rules = [
        'mes_competencia' => 'required|string',
        'ano_competencia' => 'required',
    ];

    public function updated($propertyName): void
    {
        $this->validateOnly($propertyName);
    }

    public function render()
    {
        $anos = GeradorDeAnoMesDiasHorasEMinutos::gerarAnos();
        $meses = MesEnum::cases();

        return view('livewire.balancete-financeiro.create-balancete-modal', compact('anos', 'meses'))
            ->layout('layouts.instituicao-financeira');
    }

    public function abrirCreateBalanceteModal(): void
    {
        $this->reset();
        $this->showForm = true;
    }

    public function submit(): void
    {
        $this->validate();
        sleep(1);
        $this->storeBalancete();
    }

    private function storeBalancete(): void
    {
        try {

            $this->prestador = Session::get('prestador');
            $this->prestador['instituicao'] = Session::get('instituicao');

            $balancete = [
                'mes_competencia'   => $this->mes_competencia,
                'ano_competencia'   => $this->ano_competencia,
                'historico'         => $this->historico,
                'aplica_multa'      => AplicaMultaBalanceteEnum::NAO->value,
                'prestador'         => $this->prestador,
                'prestador_id'      => $this->prestador['id'],
            ];

            (new AplicaRegrasParaCadastroBalancete())->aplicar($balancete);

            DB::beginTransaction();

            $balancete = BalancetePrestador::create($balancete);

            DB::commit();

            Log::info("Balancete Financeiro cadastrado para Pretador de ID {$this->prestador['id']}", $balancete->toArray());

            $this->showForm = false;

            $this->alert = [
                'show' => true,
                'type' => 'success',
                'title' => 'Sucesso!',
                'description' =>  "Balancete de competência " . MesEnum::from($this->mes_competencia)->name ."/{$balancete->ano_competencia} Cadastrado com sucesso."
            ];

            $this->emit('refreshDatatable');

        } catch (Exception|RuntimeException $exception) {
            DB::rollBack();
            Log::error("Cadastrando Balancete Financeiro: " .$exception->getMessage());

            $exception_error = ($exception instanceof \RuntimeException)
                ? $exception->getMessage()
                : "Houve um problema durante o procedimento e não foi possível cadastrar o balancete de competência {$this->mes_competencia}/{$this->ano_competencia}.";

            $this->alert = [
                'show' => true,
                'type' => 'danger',
                'title' => 'Erro!',
                'description' => $exception_error
            ];
        }
    }

//    /**
//     * @throws Exception
//     */
//    private function aplicarRegrasDeNegocioParaNovoCadastro() : void
//    {
//        $this->prestador = Session::get('prestador');
//        $this->prestador['instituicao'] = Session::get('instituicao');
//
//        $competenciaJaCadastrada = VerificaSeCompetenciaParaBalanceteJaEstaCadastrada::validar(
//            $this->ano_competencia,
//            (int) $this->mes_competencia,
//            $this->prestador
//        );
//
//        if ($competenciaJaCadastrada) {
//            throw new \RuntimeException("Já existem um balancete cadastrado com essa mesma competência.");
//        }
//    }
}
