<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\Lotes\SalvaLoteApi;
use App\Services\Notas\CriadorNotaFiscalServico;
use App\Services\Rps\SalvaRpsApi;
use App\Services\Xml\ConversorXmlToArray;
use App\Services\Xml\FormatadorRps;
use App\Services\Xml\XmlRetorno;
use App\Services\Xml\ValidadorLayoutXmlRequest;
use App\Validadores\LoteRps\ValidadorLotesRps;
use Bmatovu\LaravelXml\Support\XmlValidator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

/**
 * Validador XML
 * https://www.sefaz.rs.gov.br/nfe/nfe-val.aspx
 *
 */
class LoteRpsXmlCreateController extends Controller
{
    /**
     * @var XmlValidator
     */
    private $xmlValidator;

    private $lote;
    /**
     * @var XmlRetorno
     */
    private $xmlRetorno;

    public function __construct(XmlValidator $xmlValidator, XmlRetorno $xmlRetorno)
    {
        $this->xmlValidator = $xmlValidator;
        $this->lote = [];
        $this->xmlRetorno = $xmlRetorno;
    }

    public function store(Request $request)
    {
        /**
         * Validações
         */

        if (!$this->xmlValidator->is_valid($request->getContent()))
            return $this->xmlRetorno->xmlInvalido();

        $data = ConversorXmlToArray::convert($request->getContent());

        try {

            $this->lote = FormatadorRps::formatar($data);

            $errosNoLayout = ValidadorLayoutXmlRequest::validar($this->lote);

            if ($errosNoLayout)
                return $this->xmlRetorno->errorsArrayToXml($errosNoLayout);

            $lotesValidator = new ValidadorLotesRps();

            $lotesValidator->validar($this->lote);

        } catch (\Exception $e) {
            return $this->xmlRetorno->erroStringToXml($e->getMessage());
        }

     /**
      * Gravar Lote, Rps e Notas
      */

        try {
            DB::beginTransaction();

            $loteSalvo = (new SalvaLoteApi)->salvar($this->lote);

            (new SalvaRpsApi)->salvar($this->lote['rpsDoLote'], $loteSalvo->id);

            $loteSalvo->load(['rps', 'instituicao']);

            $notasFiscaisCriadas = (new CriadorNotaFiscalServico($loteSalvo))->criar();

            DB::commit();

            return $this->xmlRetorno->sucesso(
                $this->lote['Cnpj'],
                $this->lote['NumeroLote'],
                $notasFiscaisCriadas
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->xmlRetorno->erroStringToXml($e->getMessage());
        }
    }
}
